/**
* \file: SdcAuthenticator.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Android Auto
*
* \author: P. Acar / ADIT/SW2 / pacar@de.adit-jv.com
*
* \copyright (c) 2015 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <adit_logging.h>
#include <fstream>
#include <sdc_op_conv.h>
#include "SdcAuthenticator.h"

// TODO proper configuration validation

using namespace std;

LOG_IMPORT_CONTEXT(aauto_authentication)

namespace adit { namespace aauto {

SdcAuthenticator::SdcAuthenticator() {}

SdcAuthenticator::~SdcAuthenticator() {}

bool SdcAuthenticator::setCertificates(shared_ptr<GalReceiver> inReceiver)
{
    bool result = false;

    if (inReceiver == nullptr)
        return result;

    string rootCert = getRootCert();
    string clientCert = getClientCert();
    string privateKey = getPrivateKey();

    result = inReceiver->setClientCreds(rootCert, clientCert, privateKey);

    /* overwrite the stack/heap */
    memset((void*)rootCert.data(), 0, rootCert.length());
    memset((void*)clientCert.data(), 0, clientCert.length());
    memset((void*)privateKey.data(), 0, privateKey.length());

    /* necessary to avoid that the compiler removes the memset operation above */
    asm volatile("" ::: "memory");

    return result;
}

void SdcAuthenticator::setConfigItem(const string& inKey, const string& inValue)
{
    config.insert(pair<string, string>(inKey, inValue));
}

bool SdcAuthenticator::getConfigItem(const string& inKey, string& outResult)
{
    outResult = "";

    auto found = config.find(inKey);
    if (found != config.end())
    {
        outResult = found->second;
        return true;
    }

    return false;
}

string SdcAuthenticator::pullFromSecureDataContainer(const string& inFileName, uint32_t inKeyId)
{
    sdc_session_t* session = nullptr;
    sdc_wrap_unwrap_type_t* type = nullptr;
    char* unwrappedBuffer = nullptr;
    size_t unwrappedSize;

    string unwrappedStr = "";

    // create input file stream for wrapped certificate
    ifstream wrappedFile(inFileName, ios::in | ios::ate);
    if(!wrappedFile.is_open())
    {
        LOG_ERROR((aauto_authentication, "Could not open the wrapped file"));
        return unwrappedStr;
    }

    // get size of wrapped certificate and allocate a buffer for that size
    // safeguard is assumed to be 1MB
    streampos wrappedSize = wrappedFile.tellg();
    if(wrappedSize == -1 || wrappedSize > 1024*1024)
    {
        LOG_ERROR((aauto_authentication, "Could not read the wrapped file or the file is larger than expected"));
        return unwrappedStr;
    }
    char wrappedBuffer[(int)wrappedSize];

    // read file to the buffer and close file
    wrappedFile.seekg(0, ios::beg);
    wrappedFile.read(wrappedBuffer, wrappedSize);
    wrappedFile.close();

    // open SDC session
    sdc_error_t err = sdc_open_session(&session);
    if(err != SDC_OK)
    {
        LOG_ERROR((aauto_authentication, "Could not create SDC session %s", sdc_get_error_string(err)));
        return unwrappedStr;
    }

    // set the key for SDC session
    err = sdc_session_load_storage_key(session, inKeyId);
    if (err != SDC_OK)
    {
        LOG_ERROR((aauto_authentication, "Could not set SDC session key %s", sdc_get_error_string(err)));
        sdc_close_session(session);
        return unwrappedStr;
    }

    // unwrap certificate
    err = sdc_unwrap_formatted_extract_type(reinterpret_cast<unsigned char*>(wrappedBuffer), wrappedSize, &type);
    if (err != SDC_OK)
    {
        LOG_ERROR((aauto_authentication, "Could not extract type %s", sdc_get_error_string(err)));
        sdc_close_session(session);
        return unwrappedStr;
    }

    // extract type
    err = sdc_unwrap_formatted(session, type, reinterpret_cast<unsigned char*>(wrappedBuffer), wrappedSize, reinterpret_cast<unsigned char**>(&unwrappedBuffer), &unwrappedSize);
    if (err != SDC_OK)
    {
        LOG_ERROR((aauto_authentication, "Could not unwrap file %s", sdc_get_error_string(err)));
        sdc_wrap_unwrap_type_free(type);
        sdc_close_session(session);
        free(unwrappedBuffer);
        return unwrappedStr;
    }

    // free type
    sdc_wrap_unwrap_type_free(type);

    unwrappedStr.assign(unwrappedBuffer, unwrappedSize);

    // give back memory and close SDC session
    free(unwrappedBuffer);
    sdc_close_session(session);

    // return unwrapped certificate
    return unwrappedStr;
}

string SdcAuthenticator::getRootCert()
{
    string path = "";
    if(!getConfigItem("rootCertificate", path))
    {
        LOG_ERROR((aauto_authentication, "Path to root certificate was not set properly"));
        return "";
    }

    string sdcKey = "";
    if(!getConfigItem("keyId", sdcKey))
    {
        LOG_ERROR((aauto_authentication, "SDC key was not set properly"));
        return "";
    }

    return pullFromSecureDataContainer(path, stoi(sdcKey));
}

string SdcAuthenticator::getClientCert()
{
    string path = "";
    if(!getConfigItem("clientCertificate", path))
    {
        LOG_ERROR((aauto_authentication, "Path to client certificate was not set properly"));
        return "";
    }

    string sdcKey = "";
    if(!getConfigItem("keyId", sdcKey))
    {
        LOG_ERROR((aauto_authentication, "SDC key was not set properly"));
        return "";
    }

    return pullFromSecureDataContainer(path, stoi(sdcKey));
}

string SdcAuthenticator::getPrivateKey()
{
    string path = "";
    if(!getConfigItem("privateKey", path))
    {
        LOG_ERROR((aauto_authentication, "Path to private key was not set properly"));
        return "";
    }

    string sdcKey = "";
    if(!getConfigItem("keyId", sdcKey))
    {
        LOG_ERROR((aauto_authentication, "SDC key was not set properly"));
        return "";
    }

    return pullFromSecureDataContainer(path, stoi(sdcKey));
}

} } /* namespace adit { namespace aauto { */
